<?php
declare(strict_types=1);

namespace InfoZone\Http;

final class Router
{
    private array $routes = ['GET'=>[], 'POST'=>[]];

    public function get(string $path, callable|array $handler): void
    {
        $this->routes['GET'][] = ['pattern' => $this->compile($path), 'handler' => $handler];
    }

    public function post(string $path, callable|array $handler): void
    {
        $this->routes['POST'][] = ['pattern' => $this->compile($path), 'handler' => $handler];
    }

    public function dispatch(Request $req): ?Response
    {
        foreach ($this->routes[$req->method] ?? [] as $route) {
            if (preg_match($route['pattern'], $req->path, $m)) {
                $params = [];
                foreach ($m as $k => $v) if (!is_int($k)) $params[$k] = $v;
                return $this->invoke($route['handler'], $req, $params);
            }
        }
        return null;
    }

    private function compile(string $path): string
    {
        $re = preg_replace('#\{([a-zA-Z_][a-zA-Z0-9_]*)\}#', '(?P<$1>[^/]+)', $path);
        return '#^' . $re . '$#';
    }

    private function invoke(callable|array $handler, Request $req, array $params): Response
    {
        if (is_array($handler)) {
            [$class, $method] = $handler;
            $obj = new $class();
            return $obj->$method($req, $params);
        }
        return $handler($req, $params);
    }
}
